<?php

/*
Plugin Name: fawry
Description: Sity Cloud multisite site creator + points & membership integration + performance & notifications + IBAN generator + client site control.
Version: 3.0
Author: Youssef Reda Awad
*/

/**
 * تحميل ملفات اللغة
 */
add_action('plugins_loaded', 'fawry_load_textdomain');
function fawry_load_textdomain() {
    load_plugin_textdomain('fawry-site', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

/**
 * صفحة بيانات العملاء في لوحة التحكم
 */
add_action('admin_menu', 'fawry_add_admin_page');
function fawry_add_admin_page() {
    add_menu_page(
        'بيانات العملاء',
        'بيانات العملاء',
        'manage_options',
        'fawry-client-details',
        'fawry_display_client_details',
        'dashicons-id-alt',
        20
    );
}

function fawry_display_client_details() {
    echo '<div class="wrap">';
    echo '<h1>بيانات العملاء</h1>';

    $users = get_users();
    if (!$users) {
        echo '<p>لا توجد بيانات عملاء مسجلة.</p>';
        echo '</div>';
        return;
    }

    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>
        <th>اسم العميل</th>
        <th>نوع العميل</th>
        <th>اسم الشركة</th>
        <th>البريد الإلكتروني</th>
        <th>رقم الجوال</th>
        <th>الدولة</th>
        <th>المدينة</th>
        <th>رابط واتساب</th>
        <th>مجال العمل</th>
        <th>نوع النشاط</th>
        <th>طريقة التعرف على الخدمة</th>
        <th>آلية دفع النقاط</th>
        <th>وصف المشروع</th>
        <th>ميزانية التسويق</th>
        <th>وصف المنتجات والخدمات</th>
        <th>هدف الموقع</th>
        <th>رابط الموقع</th>
        <th>IBAN</th>
    </tr></thead><tbody>';

    foreach ($users as $user) {
        $client_type             = get_user_meta($user->ID, 'fawry_client_type', true);
        $company_name            = get_user_meta($user->ID, 'fawry_company_name', true);
        $phone_number            = get_user_meta($user->ID, 'fawry_phone_number', true);
        $country                 = get_user_meta($user->ID, 'fawry_country', true);
        $city                    = get_user_meta($user->ID, 'fawry_city', true);
        $whatsapp_link           = get_user_meta($user->ID, 'fawry_whatsapp_link', true);
        $industry                = get_user_meta($user->ID, 'fawry_industry', true);
        $activity_type           = get_user_meta($user->ID, 'fawry_activity_type', true);
        $referral_source         = get_user_meta($user->ID, 'fawry_referral_source', true);
        $points_payment_method   = get_user_meta($user->ID, 'fawry_points_payment_method', true);
        $project_description     = get_user_meta($user->ID, 'fawry_project_description', true);
        $marketing_budget        = get_user_meta($user->ID, 'fawry_marketing_budget', true);
        $products_description    = get_user_meta($user->ID, 'fawry_products_description', true);
        $website_goal            = get_user_meta($user->ID, 'fawry_website_goal', true);
        $site_url                = get_user_meta($user->ID, 'fawry_site_url', true);
        $iban                    = get_user_meta($user->ID, 'fawry_iban', true);

        echo '<tr>';
        echo '<td>' . esc_html($user->display_name) . '</td>';
        echo '<td>' . esc_html($client_type) . '</td>';
        echo '<td>' . esc_html($company_name) . '</td>';
        echo '<td>' . esc_html($user->user_email) . '</td>';
        echo '<td>' . esc_html($phone_number) . '</td>';
        echo '<td>' . esc_html($country) . '</td>';
        echo '<td>' . esc_html($city) . '</td>';
        echo '<td>' . ($whatsapp_link ? '<a href="' . esc_url($whatsapp_link) . '" target="_blank">واتساب</a>' : '-') . '</td>';
        echo '<td>' . esc_html($industry) . '</td>';
        echo '<td>' . esc_html($activity_type) . '</td>';
        echo '<td>' . esc_html($referral_source) . '</td>';
        echo '<td>' . esc_html($points_payment_method) . '</td>';
        echo '<td>' . esc_html($project_description) . '</td>';
        echo '<td>' . esc_html($marketing_budget) . '</td>';
        echo '<td>' . esc_html($products_description) . '</td>';
        echo '<td>' . esc_html($website_goal) . '</td>';
        echo '<td>' . ($site_url ? '<a href="' . esc_url($site_url) . '" target="_blank">' . esc_html($site_url) . '</a>' : '-') . '</td>';
        echo '<td>' . esc_html($iban) . '</td>';
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</div>';
}

/**
 * قوائم لوحة التحكم (نقاط / عضويات / جميع العملاء)
 */
add_action('admin_menu', 'fawry_add_menu_pages');
function fawry_add_menu_pages() {
    add_menu_page(
        __('لوحة نقاط العملاء', 'fawry-site'),
        __('نقاط العملاء', 'fawry-site'),
        'manage_options',
        'fawry-points-dashboard',
        'fawry_points_dashboard_page',
        'dashicons-chart-bar',
        7
    );

    add_submenu_page(
        'fawry-points-dashboard',
        __('إعدادات النقاط والباقات', 'fawry-site'),
        __('إعدادات النقاط والباقات', 'fawry-site'),
        'manage_options',
        'fawry-membership-points',
        'fawry_membership_points_page'
    );

    add_submenu_page(
        'fawry-points-dashboard',
        __('جميع العملاء', 'fawry-site'),
        __('جميع العملاء', 'fawry-site'),
        'manage_options',
        'fawry-all-clients',
        'fawry_all_clients_page'
    );

    add_submenu_page(
        'fawry-points-dashboard',
        __('بيانات المواقع', 'fawry-site'),
        __('بيانات المواقع', 'fawry-site'),
        'manage_options',
        'fawry-sites-info',
        'fawry_sites_info_page'
    );
}

/**
 * توليد IBAN للعميل (شكل داخلي فقط)
 */
function fawry_generate_iban($user_id) {
    $country  = 'EG';
    $bank     = 'SITY';
    $userPart = str_pad($user_id, 6, '0', STR_PAD_LEFT);
    $rand     = wp_rand(100000, 999999);
    return $country . '98' . $bank . $userPart . $rand;
}

/**
 * صفحة لوحة نقاط العملاء
 */
function fawry_points_dashboard_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('لوحة نقاط العملاء', 'fawry-site') . '</h1>';

    if (isset($_POST['update_points']) && isset($_POST['user_id']) && isset($_POST['new_points']) && check_admin_referer('fawry_update_points', 'fawry_points_nonce')) {
        $user_id    = intval($_POST['user_id']);
        $new_points = intval($_POST['new_points']);
        update_user_meta($user_id, 'fawry_points', $new_points);
        echo '<div class="notice notice-success is-dismissible"><p>' . __('تم تحديث النقاط بنجاح.', 'fawry-site') . '</p></div>';
    }

    $users = get_users();
    if (!$users) {
        echo '<p>' . __('لا توجد بيانات عملاء مسجلة.', 'fawry-site') . '</p>';
        echo '</div>';
        return;
    }

    echo '<table class="wp-list-table widefat fixed striped users">';
    echo '<thead><tr>
        <th>' . __('العميل', 'fawry-site') . '</th>
        <th>' . __('البريد الإلكتروني', 'fawry-site') . '</th>
        <th>' . __('عدد المواقع', 'fawry-site') . '</th>
        <th>' . __('النقاط الحالية', 'fawry-site') . '</th>
        <th>' . __('تعديل النقاط', 'fawry-site') . '</th>
    </tr></thead><tbody>';

    foreach ($users as $user) {
        $points      = get_user_meta($user->ID, 'fawry_points', true) ?: 0;
        $sites_count = fawry_get_user_sites_count($user->ID);

        echo '<tr>';
        echo '<td>' . esc_html($user->display_name) . '</td>';
        echo '<td>' . esc_html($user->user_email) . '</td>';
        echo '<td>' . esc_html($sites_count) . '</td>';
        echo '<td>' . esc_html($points) . '</td>';
        echo '<td>
            <form method="post">
                ' . wp_nonce_field('fawry_update_points', 'fawry_points_nonce', true, false) . '
                <input type="hidden" name="user_id" value="' . esc_attr($user->ID) . '">
                <input type="number" name="new_points" value="' . esc_attr($points) . '" style="width: 80px;">
                <input type="submit" name="update_points" class="button button-primary" value="' . __('تحديث', 'fawry-site') . '">
            </form>
        </td>';
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</div>';
}

function fawry_get_user_sites_count($user_id) {
    if (!is_multisite()) {
        return 0;
    }

    $sites = get_blogs_of_user($user_id);
    if (!is_array($sites)) {
        return 0;
    }

    // ID موقع الشبكة الرئيسي (Sity Cloud)
    $main_blog_id = function_exists('get_main_site_id') ? get_main_site_id() : 1;

    $count = 0;
    foreach ($sites as $site) {
        // استبعاد موقع سِتي كلاود الرئيسي من العد
        if ((int) $site->userblog_id === (int) $main_blog_id) {
            continue;
        }
        $count++;
    }

    return $count;
}


/**
 * صفحة ربط الباقات (Paid Memberships Pro) بالنقاط
 */
function fawry_membership_points_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('إعدادات النقاط والباقات', 'fawry-site') . '</h1>';
    echo '<p>' . __('يمكنك هنا تنظيم الباقات والأسعار وعدد النقاط لكل باقة عضوية في Paid Memberships Pro.', 'fawry-site') . '</p>';

    if (isset($_POST['update_level_points']) && check_admin_referer('fawry_update_level_points', 'fawry_level_nonce')) {
        $level_id = intval($_POST['level_id']);
        $points   = intval($_POST['points']);
        update_option('fawry_points_for_level_' . $level_id, $points);
        echo '<div class="updated"><p>' . __('تم تحديث النقاط بنجاح.', 'fawry-site') . '</p></div>';
    }

    if (class_exists('PMPro_Membership_Level')) {
        $levels = pmpro_getAllLevels();
        if ($levels) {
            echo '<table class="wp-list-table widefat fixed striped users">';
            echo '<thead><tr>';
            echo '<th>' . __('اسم العضوية', 'fawry-site') . '</th>';
            echo '<th>' . __('عدد النقاط', 'fawry-site') . '</th>';
            echo '</tr></thead><tbody>';

            foreach ($levels as $level) {
                $level_points = get_option('fawry_points_for_level_' . $level->id, 0);
                echo '<tr>';
                echo '<td>' . esc_html($level->name) . '</td>';
                echo '<td>
                    <form method="POST" action="">
                        ' . wp_nonce_field('fawry_update_level_points', 'fawry_level_nonce', true, false) . '
                        <input type="hidden" name="level_id" value="' . esc_attr($level->id) . '">
                        <input type="number" name="points" value="' . esc_attr($level_points) . '" style="width: 80px;">
                        <input type="submit" name="update_level_points" value="' . esc_attr__('تحديث', 'fawry-site') . '" class="button">
                    </form>
                </td>';
                echo '</tr>';
            }

            echo '</tbody></table>';
        } else {
            echo '<p>' . __('لا توجد مستويات عضوية حتى الآن.', 'fawry-site') . '</p>';
        }
    } else {
        echo '<p>' . __('لم يتم تثبيت Paid Memberships Pro بشكل صحيح.', 'fawry-site') . '</p>';
    }

    echo '</div>';
}

/**
 * عند تغيير مستوى العضوية، حدّث نقاط العميل تلقائياً
 */
add_action('pmpro_after_change_membership_level', 'fawry_upgrade_points_based_on_membership', 10, 2);
function fawry_upgrade_points_based_on_membership($level_id, $user_id) {
    $points = get_option('fawry_points_for_level_' . $level_id, 0);
    update_user_meta($user_id, 'fawry_points', $points);
}

/**
 * صفحة جميع العملاء + توليد IBAN
 */
function fawry_all_clients_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('جميع العملاء', 'fawry-site') . '</h1>';

    if (isset($_POST['generate_iban']) && isset($_POST['user_id']) && check_admin_referer('fawry_generate_iban', 'fawry_iban_nonce')) {
        $user_id = intval($_POST['user_id']);
        $iban    = fawry_generate_iban($user_id);
        update_user_meta($user_id, 'fawry_iban', $iban);
        echo '<div class="notice notice-success is-dismissible"><p>' . __('تم توليد IBAN للعميل بنجاح.', 'fawry-site') . '</p></div>';
    }

    $users = get_users();

    echo '<table class="wp-list-table widefat fixed striped users">';
    echo '<thead><tr>
        <th>' . __('العميل', 'fawry-site') . '</th>
        <th>' . __('البريد الإلكتروني', 'fawry-site') . '</th>
        <th>' . __('عدد المواقع', 'fawry-site') . '</th>
        <th>' . __('النقاط الحالية', 'fawry-site') . '</th>
        <th>' . __('IBAN', 'fawry-site') . '</th>
        <th>' . __('إدارة IBAN', 'fawry-site') . '</th>
    </tr></thead><tbody>';

    foreach ($users as $user) {
        $points      = get_user_meta($user->ID, 'fawry_points', true) ?: 0;
        $sites_count = fawry_get_user_sites_count($user->ID);
        $iban        = get_user_meta($user->ID, 'fawry_iban', true);

        echo '<tr>';
        echo '<td>' . esc_html($user->display_name) . '</td>';
        echo '<td>' . esc_html($user->user_email) . '</td>';
        echo '<td>' . esc_html($sites_count) . '</td>';
        echo '<td>' . esc_html($points) . '</td>';
        echo '<td>' . esc_html($iban) . '</td>';
        echo '<td>
            <form method="post">
                ' . wp_nonce_field('fawry_generate_iban', 'fawry_iban_nonce', true, false) . '
                <input type="hidden" name="user_id" value="' . esc_attr($user->ID) . '">
                <input type="submit" name="generate_iban" class="button" value="' . esc_attr__('توليد / إعادة توليد IBAN', 'fawry-site') . '">
            </form>
        </td>';
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</div>';
}


/**
 * صفحة "بيانات المواقع" في لوحة تحكم الأدمن – مع إمكانية التصدير كملف CSV (Excel / Google Sheets)
 */
function fawry_sites_info_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('غير مسموح لك بالوصول إلى هذه الصفحة.', 'fawry-site'));
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('بيانات المواقع المنشأة من Sity Cloud', 'fawry-site') . '</h1>';

    global $wpdb;
    $table_name = $wpdb->base_prefix . 'sitycloud_sites_info';

    // تصدير جميع البيانات كملف CSV يمكن فتحه في Excel أو Google Sheets
    if (isset($_GET['export']) && $_GET['export'] === 'all') {
        $rows = $wpdb->get_results("SELECT * FROM {$table_name} ORDER BY created_at DESC", ARRAY_A);
        if (!is_array($rows)) {
            $rows = array();
        }

        $filename = 'sitycloud-sites-' . date('Y-m-d') . '.csv';

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=' . $filename);

        $output = fopen('php://output', 'w');

        if (!empty($rows)) {
            fputcsv($output, array_keys($rows[0]));
            foreach ($rows as $row) {
                fputcsv($output, $row);
            }
        } else {
            fputcsv($output, array('لا توجد بيانات مواقع مسجلة حتى الآن.'));
        }

        fclose($output);
        exit;
    }

    $rows = $wpdb->get_results("SELECT * FROM {$table_name} ORDER BY created_at DESC LIMIT 500", ARRAY_A);

    echo '<p>' . esc_html__('هنا تجد ملخصًا مركزيًا لكل موقع تم إنشاؤه (تعريف المشروع، هدف الموقع، طريقة العمل...).', 'fawry-site') . '</p>';
    echo '<p><a href="' . esc_url(add_query_arg('export', 'all')) . '" class="button button-primary">' .
        esc_html__('تصدير جميع المواقع (CSV)', 'fawry-site') . '</a></p>';

    if (empty($rows)) {
        echo '<p>' . esc_html__('لا توجد مواقع مسجلة حتى الآن.', 'fawry-site') . '</p>';
        echo '</div>';
        return;
    }

    $columns = array(
        'id'                          => 'ID',
        'site_id'                     => 'ID الموقع',
        'site_url'                    => 'رابط الموقع',
        'site_name'                   => 'اسم الموقع',
        'user_id'                     => 'ID العميل',
        'phone_number'                => 'الهاتف',
        'country'                     => 'الدولة',
        'city'                        => 'المدينة',
        'whatsapp_link'               => 'واتساب',
        'client_type'                 => 'نوع العميل',
        'company_name'                => 'اسم الشركة',
        'business_registration_number'=> 'السجل التجاري',
        'industry'                    => 'النشاط',
        'activity_type'               => 'طريقة العمل',
        'referral_source'             => 'مصدر معرفتنا',
        'points_payment_method'       => 'طريقة دفع النقاط',
        'project_description'         => 'تعريف المشروع',
        'marketing_budget'           => 'ميزانية التسويق',
        'products_description'       => 'المنتجات / الخدمات',
        'website_goal'               => 'هدف الموقع',
        'created_at'                 => 'تاريخ الإنشاء',
    );

    echo '<table class="wp-list-table widefat fixed striped">';
    echo '<thead><tr>';
    foreach ($columns as $key => $label) {
        echo '<th>' . esc_html($label) . '</th>';
    }
    echo '</tr></thead><tbody>';

    foreach ($rows as $row) {
        echo '<tr>';
        foreach ($columns as $key => $label) {
            $value = isset($row[$key]) ? $row[$key] : '';
            if ($key === 'site_url' && !empty($value)) {
                $value = '<a href="' . esc_url($value) . '" target="_blank" rel="noopener noreferrer">' . esc_html($value) . '</a>';
            } elseif (in_array($key, array('project_description', 'products_description', 'website_goal'), true)) {
                $value = wp_trim_words(wp_strip_all_tags($value), 18, '...');
            } else {
                $value = esc_html($value);
            }
            echo '<td>' . $value . '</td>';
        }
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</div>';
}



/**
 * شورت كود عرض نقاط المستخدم
 */
add_shortcode('fawry_display_user_points', 'fawry_display_user_points_shortcode');
function fawry_display_user_points_shortcode() {
    if (!is_user_logged_in()) {
        return __('يجب تسجيل الدخول لعرض النقاط.', 'fawry-site');
    }

    $current_user = wp_get_current_user();
    $points       = get_user_meta($current_user->ID, 'fawry_points', true) ?: 0;
    $message      = '';

    if (isset($_GET['points_deducted'])) {
        $message = sprintf(__('تم خصم %d نقاط لإنشاء موقع جديد.', 'fawry-site'), intval($_GET['points_deducted']));
    } elseif (isset($_GET['points_added'])) {
        $message = sprintf(__('تم إضافة %d نقاط.', 'fawry-site'), intval($_GET['points_added']));
    }

    return sprintf(__('لديك %d نقاط.', 'fawry-site'), $points) . ' ' . esc_html($message);
}

/**
 * زر إنشاء موقع جديد
 */
add_shortcode('fawry_create_site_button_new', 'fawry_create_site_button_new_shortcode');
function fawry_create_site_button_new_shortcode() {
    return '<a href="https://sity.cloud/websub" class="button1">' . __('إنشاء موقع جديد', 'fawry-site') . '</a>';
}

/**
 * شورت كود بيانات بسيطة للمستخدم
 */
function fawry_display_user_name() {
    $current_user = wp_get_current_user();
    return ($current_user && $current_user->ID) ? esc_html($current_user->display_name) : esc_html__('ضيف', 'fawry-site');
}
add_shortcode('fawry_display_user_name', 'fawry_display_user_name');

function fawry_display_user_email() {
    $current_user = wp_get_current_user();
    return ($current_user && $current_user->user_email) ? esc_html($current_user->user_email) : '';
}
add_shortcode('fawry_display_user_email', 'fawry_display_user_email');

function fawry_display_user_id() {
    $current_user = wp_get_current_user();
    return ($current_user && $current_user->ID) ? intval($current_user->ID) : '';
}
add_shortcode('fawry_display_user_id', 'fawry_display_user_id');

function fawry_display_user_iban() {
    $current_user = wp_get_current_user();
    $iban         = get_user_meta($current_user->ID, 'fawry_iban', true);
    return esc_html($iban);
}
add_shortcode('fawry_display_user_iban', 'fawry_display_user_iban');

/**
 * شورت كود الفورم – wizard من 3 خطوات بتصميم سِتي كلاود
 */
add_shortcode('fawry_site_creation_form', 'fawry_site_creation_form_shortcode');
function fawry_site_creation_form_shortcode() {
    if (!is_user_logged_in()) {
        return '<p>' . __('يجب تسجيل الدخول لإنشاء موقع جديد.', 'fawry-site') . '</p>';
    }

    ob_start();

    if (isset($_GET['error'])) {
        $msg = '';
        if ($_GET['error'] === 'subdirectory_exists') {
            $msg = 'عذرًا، هذا الرابط مأخوذ بالفعل. جرّب رابط آخر مميز لبراندك.';
        } elseif ($_GET['error'] === 'insufficient_points') {
            $msg = 'رصيد نقاطك غير كافٍ لإنشاء موقع جديد (تكلفة إنشاء الموقع 10 نقاط). برجاء ترقية باقتك أو شحن النقاط.';
        } elseif ($_GET['error'] === 'site_creation_failed') {
            $msg = 'حدث خطأ أثناء إنشاء الموقع. برجاء المحاولة مرة أخرى أو التواصل مع الدعم.';
        }
        if ($msg) {
            echo '<div class="sity-alert sity-alert-error">' . esc_html($msg) . '</div>';
            if ($_GET['error'] === 'insufficient_points') {
                ?>
                <script>
                document.addEventListener('DOMContentLoaded', function() {
                    alert('رصيد نقاطك غير كافٍ لإنشاء موقع جديد على Sity Cloud.\n\nاشترك في باقة مدفوعة أو جدد اشتراكك لشحن النقاط والمتابعة في إنشاء مواقعك.');
                });
                </script>
                <?php
            }
        }
    }
    ?>

    <form class="form_container" id="fawry-create-site-form" method="POST" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <input type="hidden" name="action" value="fawry_create_site">

        <div class="title_container">
            <div>
                <p class="title">إنشاء موقع جديد على Sity Cloud</p>
                <span class="subtitle">8–10 دقائق ويكون موقعك جاهز على الإنترنت – تكلّفة الإنشاء: 10 نقاط من رصيدك</span>
            </div>
            <button type="button" class="sity-notification-bell" onclick="SityCloudShowNotification('سيتم إرسال إشعارات مهمة عن مواقعك ونقاطك هنا 🔔','info')" aria-label="الإشعارات"></button>
        </div>

        <div class="sity-step-indicator">
            <div class="sity-step-indicator-item sity-step-active">
                <span class="sity-step-number">1</span>
                <span class="sity-step-label">بيانات الموقع</span>
            </div>
            <div class="sity-step-indicator-item">
                <span class="sity-step-number">2</span>
                <span class="sity-step-label">بيانات العميل</span>
            </div>
            <div class="sity-step-indicator-item">
                <span class="sity-step-number">3</span>
                <span class="sity-step-label">تفاصيل المشروع</span>
            </div>
        </div>

        <div class="sity-step" data-step="1">
            <p class="sity-step-info">ابدأ باختيار اسم يعبر عن البراند ورابط مميز لموقعك.</p>

            <div class="input_container">
                <label class="input_label" for="site_name">اسم الموقع</label>
                <input placeholder="مثال: متجر Top Edu أو Sity Studio" type="text" class="input_field" name="site_name" id="site_name" data-required="1">
            </div>

            <div class="input_container">
                <label class="input_label" for="subdirectory">رابط الموقع (URL slug)</label>
                <div class="input_field_container">
                    <span class="url_prefix">sity.cloud/</span>
                    <input placeholder="اكتب الرابط بالإنجليزية – بدون مسافات" type="text" class="input_field" name="subdirectory" id="subdirectory" data-required="1" pattern="[a-z0-9\-]+">
                </div>
                <small class="field_hint">يُفضّل أن يكون قصير، بحروف إنجليزية صغيرة فقط، بدون مسافات، مثل: myshop أو mybrand.</small>
            </div>
        </div>

        <div class="sity-step" data-step="2" style="display:none;">
            <p class="sity-step-info">عرّفنا بيك… هل أنت فرد أم شركة؟ وطرق التواصل الأساسية.</p>

            <div class="input_container">
                <label class="input_label">نوع العميل</label>
                <div class="radio_container">
                    <label class="radio_option">
                        <input type="radio" name="client_type" value="individual" onclick="toggleCompanyFields()">
                        <span>فرد</span>
                    </label>
                    <label class="radio_option">
                        <input type="radio" name="client_type" value="company" onclick="toggleCompanyFields()">
                        <span>شركة</span>
                    </label>
                </div>
            </div>

            <div class="sity-step-grid-2">
                <div class="input_container" id="company_name_container" style="display:none;">
                    <label class="input_label" for="company_name">اسم الشركة</label>
                    <input placeholder="مثال: Sity Cloud Technologies" type="text" class="input_field" name="company_name" id="company_name">
                </div>

                <div class="input_container" id="business_registration_number_container" style="display:none;">
                    <label class="input_label" for="business_registration_number">رقم السجل التجاري</label>
                    <input placeholder="مثال: 1234567890" type="text" class="input_field" name="business_registration_number" id="business_registration_number">
                </div>
            </div>

            <div class="sity-step-grid-2">
                <div class="input_container">
                    <label class="input_label" for="phone_number">رقم الجوال</label>
                    <input placeholder="مثال: 2011XXXXXXX+" type="tel" class="input_field" name="phone_number" id="phone_number" data-required="1">
                </div>

                <div class="input_container">
                    <label class="input_label" for="whatsapp_link">رابط واتساب (اختياري)</label>
                    <input placeholder="https://wa.me/2011XXXXXXX" type="url" class="input_field" name="whatsapp_link" id="whatsapp_link">
                </div>
            </div>

            <div class="sity-step-grid-2">
                <div class="input_container">
                    <label class="input_label" for="country">الدولة</label>
                    <input placeholder="مثال: مصر / السعودية" type="text" class="input_field" name="country" id="country" data-required="1">
                </div>
                <div class="input_container">
                    <label class="input_label" for="city">المدينة</label>
                    <input placeholder="مثال: القاهرة / الرياض" type="text" class="input_field" name="city" id="city" data-required="1">
                </div>
            </div>
        </div>

        <div class="sity-step" data-step="3" style="display:none;">
            <p class="sity-step-info">عرفنا أكثر عن مشروعك، هدفك من الموقع، وطريقة عملك.</p>

            <div class="sity-step-grid-2">
                <div class="input_container">
                    <label class="input_label" for="industry">مجال العمل</label>
                    <select name="industry" id="industry" class="input_field" data-required="1">
                        <option value="">اختر المجال...</option>
                        <option value="technology">تكنولوجيا</option>
                        <option value="education">تعليم</option>
                        <option value="healthcare">رعاية صحية</option>
                        <option value="retail">تجزئة</option>
                        <option value="services">خدمات</option>
                        <option value="other">أخرى</option>
                    </select>
                </div>

                <div class="input_container">
                    <label class="input_label" for="activity_type">نوع النشاط</label>
                    <select name="activity_type" id="activity_type" class="input_field" data-required="1">
                        <option value="">اختر نوع النشاط...</option>
                        <option value="online">نشاط إلكتروني</option>
                        <option value="offline">نشاط على أرض الواقع</option>
                        <option value="both">الإثنان</option>
                    </select>
                </div>
            </div>

            <div class="sity-step-grid-2">
                <div class="input_container">
                    <label class="input_label" for="referral_source">كيف سمعت عن Sity Cloud؟</label>
                    <select name="referral_source" id="referral_source" class="input_field">
                        <option value="search">بحث في جوجل</option>
                        <option value="social">سوشيال ميديا</option>
                        <option value="friend">ترشيح من صديق</option>
                        <option value="ads">إعلان ممول</option>
                        <option value="other">أخرى</option>
                    </select>
                </div>

                <div class="input_container">
                    <label class="input_label" for="points_payment_method">آلية دفع النقاط</label>
                    <select name="points_payment_method" id="points_payment_method" class="input_field">
                        <option value="online">دفع إلكتروني</option>
                        <option value="bank_transfer">تحويل بنكي</option>
                        <option value="wallet">محفظة إلكترونية</option>
                    </select>
                </div>
            </div>

            <div class="input_container">
                <label class="input_label" for="project_description">وصف المشروع</label>
                <textarea class="input_field" name="project_description" id="project_description" rows="3" data-required="1" placeholder="اشرح طبيعة مشروعك والبراند اللي هتقدمه على الموقع."></textarea>
            </div>

            <div class="sity-step-grid-2">
                <div class="input_container">
                    <label class="input_label" for="marketing_budget">ميزانية التسويق الشهرية (تقريبية)</label>
                    <input placeholder="مثال: 5000 جنيه" type="text" class="input_field" name="marketing_budget" id="marketing_budget">
                </div>

                <div class="input_container">
                    <label class="input_label" for="products_description">وصف المنتجات / الخدمات</label>
                    <textarea class="input_field" name="products_description" id="products_description" rows="2" placeholder="اذكر أهم المنتجات أو الخدمات اللي بتقدمها."></textarea>
                </div>
            </div>

            <div class="input_container">
                <label class="input_label" for="website_goal">هدف الموقع</label>
                <textarea class="input_field" name="website_goal" id="website_goal" rows="3" data-required="1" placeholder="بيع – تعريف بالخدمة – حجز مواعيد – تجميع ليدز..."></textarea>
            </div>

            <div class="checkbox_container">
                <input type="checkbox" name="terms" id="terms" data-required="1">
                <label for="terms">أوافق على <a href="https://sity.cloud/terms" target="_blank">الشروط والأحكام</a> و
                    <a href="https://sity.cloud/privacy" target="_blank">سياسة الخصوصية</a>.
                </label>
            </div>
        </div>

        <div class="wizard_buttons">
            <button type="button" class="button1 sity-step-prev" style="display:none;"><span>←</span><span>السابق</span></button>
            <div class="wizard_buttons_right">
                <button type="button" class="button1 sity-step-next"><span>التالي</span><span>→</span></button>
                <button type="submit" class="button1 sity-step-submit" style="display:none;">
                    <span>⚡</span><span>ابدأ إنشاء موقعك الآن</span>
                </button>
            </div>
        </div>
    </form>

    <script>
    function toggleCompanyFields() {
        var companyRadio = document.querySelector('input[name="client_type"][value="company"]');
        var companyName  = document.getElementById('company_name_container');
        var regNumber    = document.getElementById('business_registration_number_container');
        if (!companyRadio || !companyName || !regNumber) return;
        if (companyRadio.checked) {
            companyName.style.display = 'flex';
            regNumber.style.display   = 'flex';
        } else {
            companyName.style.display = 'none';
            regNumber.style.display   = 'none';
        }
    }
    document.addEventListener('DOMContentLoaded', toggleCompanyFields);
    </script>

    <?php
    return ob_get_clean();
}

/**
 * معالجة إنشاء الموقع – تكلفة 10 نقاط
 */
add_action('admin_post_fawry_create_site', 'fawry_create_site_handler');
add_action('admin_post_nopriv_fawry_create_site', 'fawry_create_site_handler');

function fawry_create_site_handler() {
    if (!is_user_logged_in()) {
        wp_redirect(wp_login_url());
        exit;
    }

    $current_user = wp_get_current_user();

    $required_fields = array('site_name', 'subdirectory');
    foreach ($required_fields as $field) {
        if (empty($_POST[$field])) {
            wp_redirect(add_query_arg('error', 'site_creation_failed', 'https://sity.cloud/websub'));
            exit;
        }
    }

    $site_name    = sanitize_text_field($_POST['site_name']);
    $subdirectory = sanitize_title($_POST['subdirectory']);

    $phone_number              = isset($_POST['phone_number']) ? sanitize_text_field($_POST['phone_number']) : '';
    $country                   = isset($_POST['country']) ? sanitize_text_field($_POST['country']) : '';
    $city                      = isset($_POST['city']) ? sanitize_text_field($_POST['city']) : '';
    $whatsapp_link             = isset($_POST['whatsapp_link']) ? esc_url_raw($_POST['whatsapp_link']) : '';
    $client_type               = isset($_POST['client_type']) ? sanitize_text_field($_POST['client_type']) : 'individual';
    $company_name              = isset($_POST['company_name']) ? sanitize_text_field($_POST['company_name']) : '';
    $business_registration_num = isset($_POST['business_registration_number']) ? sanitize_text_field($_POST['business_registration_number']) : '';
    $industry                  = isset($_POST['industry']) ? sanitize_text_field($_POST['industry']) : '';
    $activity_type             = isset($_POST['activity_type']) ? sanitize_text_field($_POST['activity_type']) : '';
    $referral_source           = isset($_POST['referral_source']) ? sanitize_text_field($_POST['referral_source']) : '';
    $points_payment_method     = isset($_POST['points_payment_method']) ? sanitize_text_field($_POST['points_payment_method']) : '';
    $project_description       = isset($_POST['project_description']) ? sanitize_textarea_field($_POST['project_description']) : '';
    $marketing_budget          = isset($_POST['marketing_budget']) ? sanitize_text_field($_POST['marketing_budget']) : '';
    $products_description      = isset($_POST['products_description']) ? sanitize_textarea_field($_POST['products_description']) : '';
    $website_goal              = isset($_POST['website_goal']) ? sanitize_textarea_field($_POST['website_goal']) : '';

    if (is_subdomain_install()) {
        $newdomain = $subdirectory . '.' . preg_replace('|^www\.|', '', $_SERVER['HTTP_HOST']);
        $path      = '/';
    } else {
        $newdomain = preg_replace('|^www\.|', '', $_SERVER['HTTP_HOST']);
        $path      = '/' . $subdirectory . '/';
    }

    if (domain_exists($newdomain, $path, 1)) {
        wp_redirect(add_query_arg('error', 'subdirectory_exists', 'https://sity.cloud/websub'));
        exit;
    }

    $current_points = get_user_meta($current_user->ID, 'fawry_points', true) ?: 0;
    $site_cost      = 10;
    if ($current_points < $site_cost) {
        wp_redirect(add_query_arg('error', 'insufficient_points', 'https://sity.cloud/websub'));
        exit;
    }

    $new_points = $current_points - $site_cost;
    update_user_meta($current_user->ID, 'fawry_points', $new_points);

    $user_id = $current_user->ID;
    $meta    = array(
        'public'          => 1,
        'blogdescription' => $site_name,
    );

    $blog_id = wpmu_create_blog($newdomain, $path, $site_name, $user_id, $meta, get_current_network_id());

    if (is_wp_error($blog_id)) {
        wp_redirect(add_query_arg('error', 'site_creation_failed', 'https://sity.cloud/websub'));
        exit;
    }

    // حفظ رابط الموقع الجديد في بيانات العميل
    $new_site_url = get_home_url($blog_id, '/');
    update_user_meta($user_id, 'fawry_site_url', $new_site_url);

    // تسجيل بيانات المشروع / الهدف / طريقة العمل في جدول مركزي على مستوى الشبكة
    fawry_store_site_project_info(
        $blog_id,
        $user_id,
        array(
            'site_url'                     => $new_site_url,
            'site_name'                    => $site_name,
            'phone_number'                 => $phone_number,
            'country'                      => $country,
            'city'                         => $city,
            'whatsapp_link'                => $whatsapp_link,
            'client_type'                  => $client_type,
            'company_name'                 => $company_name,
            'business_registration_number' => $business_registration_num,
            'industry'                     => $industry,
            'activity_type'                => $activity_type,
            'referral_source'              => $referral_source,
            'points_payment_method'        => $points_payment_method,
            'project_description'          => $project_description,
            'marketing_budget'             => $marketing_budget,
            'products_description'         => $products_description,
            'website_goal'                 => $website_goal,
        )
    );

    update_user_meta($user_id, 'fawry_phone_number', $phone_number);
    update_user_meta($user_id, 'fawry_country', $country);
    update_user_meta($user_id, 'fawry_city', $city);
    update_user_meta($user_id, 'fawry_whatsapp_link', $whatsapp_link);
    update_user_meta($user_id, 'fawry_client_type', $client_type);
    update_user_meta($user_id, 'fawry_company_name', $company_name);
    update_user_meta($user_id, 'fawry_business_registration_number', $business_registration_num);
    update_user_meta($user_id, 'fawry_industry', $industry);
    update_user_meta($user_id, 'fawry_activity_type', $activity_type);
    update_user_meta($user_id, 'fawry_referral_source', $referral_source);
    update_user_meta($user_id, 'fawry_points_payment_method', $points_payment_method);
    update_user_meta($user_id, 'fawry_project_description', $project_description);
    update_user_meta($user_id, 'fawry_marketing_budget', $marketing_budget);
    update_user_meta($user_id, 'fawry_products_description', $products_description);
    update_user_meta($user_id, 'fawry_website_goal', $website_goal);

    $existing_iban = get_user_meta($user_id, 'fawry_iban', true);
    if (!$existing_iban) {
        $iban = fawry_generate_iban($user_id);
        update_user_meta($user_id, 'fawry_iban', $iban);
    }

    $new_site_url = get_home_url($blog_id, '/');
    update_user_meta($user_id, 'fawry_site_url', $new_site_url);

    $success_url = add_query_arg(
        array(
            'site_created'    => 1,
            'site_url'        => rawurlencode($new_site_url),
            'points_deducted' => $site_cost,
        ),
        'https://sity.cloud/webcreated'
    );
    wp_redirect($success_url);
    exit;
}

/**
 * هاندلر إدارة المواقع (تعليق / تشغيل / حذف)
 */
add_action('admin_post_fawry_manage_site', 'fawry_manage_site_handler');
add_action('admin_post_nopriv_fawry_manage_site', 'fawry_manage_site_handler');
function fawry_manage_site_handler() {
    if (!is_user_logged_in() || !is_multisite()) {
        wp_redirect(home_url());
        exit;
    }

    if (!isset($_POST['fawry_manage_site_nonce']) || !wp_verify_nonce($_POST['fawry_manage_site_nonce'], 'fawry_manage_site')) {
        wp_redirect(add_query_arg('site_action_result', 'error', 'https://sity.cloud/mysites'));
        exit;
    }

    $blog_id = isset($_POST['blog_id']) ? intval($_POST['blog_id']) : 0;
    $action  = isset($_POST['site_action']) ? sanitize_text_field($_POST['site_action']) : '';

    if (!$blog_id || !$action) {
        wp_redirect(add_query_arg('site_action_result', 'error', 'https://sity.cloud/mysites'));
        exit;
    }

    $current_user = wp_get_current_user();
    $user_id      = $current_user->ID;

    $has_access = is_super_admin($user_id) || is_user_member_of_blog($user_id, $blog_id);
    if (!$has_access) {
        wp_redirect(add_query_arg('site_action_result', 'no_permission', 'https://sity.cloud/mysites'));
        exit;
    }

    $result = 'error';

    if ($action === 'pause') {
        if (function_exists('update_blog_details')) {
            update_blog_details($blog_id, array('public' => 0));
        }
        update_blog_option($blog_id, 'sitycloud_site_status', 'paused');
        $result = 'paused';
    } elseif ($action === 'resume') {
        if (function_exists('update_blog_details')) {
            update_blog_details($blog_id, array('public' => 1));
        }
        update_blog_option($blog_id, 'sitycloud_site_status', 'active');
        $result = 'resumed';
    } elseif ($action === 'delete') {
        if (is_super_admin($user_id)) {
            wpmu_delete_blog($blog_id, true);
            $result = 'deleted';
        } else {
            update_blog_option($blog_id, 'sitycloud_site_status', 'delete_requested');
            $admin_email = get_site_option('admin_email');
            $site_url    = get_home_url($blog_id, '/');
            if ($admin_email) {
                $subject = 'طلب حذف موقع من عميل Sity Cloud';
                $message = "العميل {$current_user->user_email} طلب حذف موقعه: {$site_url} (Blog ID: {$blog_id})";
                wp_mail($admin_email, $subject, $message);
            }
            $result = 'delete_requested';
        }
    }

    wp_redirect(add_query_arg('site_action_result', $result, 'https://sity.cloud/mysites'));
    exit;
}

/**
 * شورت كود عرض المواقع في شكل كروت + سكرين شوت + زر لوحة التحكم (نافذة منبثقة فول سكرين)
 */
add_shortcode('fawry_display_sites_custom', 'fawry_display_sites_custom_shortcode');
function fawry_display_sites_custom_shortcode() {
    if (!is_user_logged_in() || !is_multisite()) {
        return __('يجب تسجيل الدخول على شبكة Sity Cloud.', 'fawry-site');
    }

    $current_user = wp_get_current_user();
    $sites        = get_blogs_of_user($current_user->ID);
    $main_blog_id = function_exists('get_main_site_id') ? get_main_site_id() : 1;
    $iban         = get_user_meta($current_user->ID, 'fawry_iban', true);

    if (empty($sites)) {
        return '<p>' . __('لا توجد مواقع مرتبطة بحسابك حتى الآن.', 'fawry-site') . '</p>';
    }

    ob_start();
    echo '<div class="fawry-cards">';
    foreach ($sites as $site) {

    // تجاهل موقع سِتي كلاود الرئيسي
    if ((int) $site->userblog_id === (int) $main_blog_id) {
        continue;
    }

    $blog_id   = $site->userblog_id;
        $url       = get_home_url($blog_id, '/');
        $name      = get_blog_option($blog_id, 'blogname');
        $siteObj   = get_site($blog_id);
        $created   = $siteObj && !empty($siteObj->registered) ? mysql2date('Y-m-d', $siteObj->registered) : '';
        $score     = get_blog_option($blog_id, 'sitycloud_last_performance_score', '');
        $status    = get_blog_option($blog_id, 'sitycloud_site_status', 'active');

        // Screenshot مخصص من براند Sity Cloud (بدون شعار ووردبريس)
        $screenshot_url = plugin_dir_url(__FILE__) . 'assets/sitycloud-site-placeholder.png';

        // رابط لوحة التحكم المباشر لموقع العميل
        $admin_url = get_admin_url($blog_id);

        echo '<div class="fawry-card">';

        // Screenshot
        echo '<div class="fawry-card-thumb">';
        echo '<img src="' . esc_url($screenshot_url) . '" alt="' . esc_attr($name) . '" loading="lazy" />';
        echo '</div>';

        echo '<div class="fawry-card-inner">';

        echo '<div class="fawry-card-header-row">';
        $status_label = 'موقع فعال';
        $status_class = 'fawry-status-pill-active';
        if ($status === 'paused') {
            $status_label = 'موقع مُعلّق';
            $status_class = 'fawry-status-pill-paused';
        } elseif ($status === 'delete_requested') {
            $status_label = 'في انتظار حذف من الدعم';
            $status_class = 'fawry-status-pill-delete';
        }

        echo '<span class="fawry-status-pill ' . esc_attr($status_class) . '">' . esc_html($status_label) . '</span>';
        if ($iban) {
            echo '<span class="fawry-iban-pill" title="IBAN الخاص بحسابك">' . esc_html($iban) . '</span>';
        }
        echo '</div>';

        echo '<h3><a href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($name) . '</a></h3>';
        echo '<div class="fawry-card-meta">' . esc_html($url) . '</div>';

        echo '<div class="fawry-card-chips">';
        if ($created) {
            echo '<span class="fawry-chip">تاريخ الإنشاء: ' . esc_html($created) . '</span>';
        }
        if ($score !== '') {
            echo '<span class="fawry-chip">أداء تقريبي: ' . intval($score) . '/100</span>';
        }
        echo '<span class="fawry-chip fawry-chip-cost">تكلفة الإنشاء: 10 نقاط</span>';
        echo '</div>';

        echo '<div class="fawry-card-actions">';
        echo '<a class="button1" href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer"><span>↗</span><span>زيارة الموقع</span></a>';
        echo '<a class="button1" href="' . esc_url($admin_url) . '" target="_blank" rel="noopener noreferrer"><span>⚙</span><span>لوحة التحكم</span></a>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="fawry-card-form">';
        echo wp_nonce_field('fawry_manage_site', 'fawry_manage_site_nonce', true, false);
        echo '<input type="hidden" name="action" value="fawry_manage_site">';
        echo '<input type="hidden" name="blog_id" value="' . esc_attr($blog_id) . '">';

        if ($status === 'paused') {
            echo '<button type="submit" class="button1 fawry-site-action fawry-site-resume" name="site_action" value="resume"><span>▶</span><span>تشغيل الموقع</span></button>';
        } else {
            echo '<button type="submit" class="button1 fawry-site-action fawry-site-pause" name="site_action" value="pause"><span>⏸</span><span>تعليق الموقع</span></button>';
        }

        echo '<button type="submit" class="button1 fawry-site-action fawry-site-delete" name="site_action" value="delete"><span>🗑</span><span>حذف الموقع</span></button>';

        echo '</form>';

        echo '</div>'; // actions
        echo '</div>'; // inner
        echo '</div>'; // card
    }
    echo '</div>';

    return ob_get_clean();
}

/**
 * صفحة النجاح بعد إنشاء الموقع (مع إعادة توجيه تلقائي إلى /mysites بعد 5 ثواني)
 */
add_shortcode('fawry_successfully_page', 'fawry_successfully_page_shortcode');
function fawry_successfully_page_shortcode() {
    if (!isset($_GET['site_created'])) {
        wp_redirect('https://sity.cloud/dashboard/');
        exit;
    }

    $site_url        = isset($_GET['site_url']) ? esc_url_raw($_GET['site_url']) : '';
    $points_deducted = isset($_GET['points_deducted']) ? intval($_GET['points_deducted']) : 0;

    ob_start();
    ?>
    <div id="success-page">
        <div class="title_container">
            <div>
                <h1>تم إنشاء موقعك بنجاح!</h1>
                <div class="subtitle">مبروك 🎉 – موقعك على Sity Cloud جاهز دلوقتي للتخصيص والانطلاق</div>
                <div class="subtitle" style="margin-top:4px;">سيتم تحويلك تلقائيًا إلى صفحة مواقعي خلال ثوانٍ...</div>
            </div>
        </div>

        <div class="terminal-loader">
            <div class="terminal-header">
                <div class="terminal-title">Sity Cloud • Site Engine</div>
                <div class="terminal-buttons"><span></span><span></span><span></span></div>
            </div>
            <div class="terminal-body">
                <span class="terminal-line"><span class="terminal-line-prefix">$</span> إطلاق موقع جديد...</span>
                <span class="terminal-line terminal-line-success">✔ تم تجهيز الاستضافة وضبط الإعدادات الأساسية</span>
                <span class="terminal-line terminal-line-success">✔ تم ربط حسابك بنظام النقاط والعضويات في Sity Cloud</span>
                <?php if ($site_url): ?>
                    <span class="terminal-line"><span class="terminal-line-prefix">$</span> عنوان موقعك: <span class="terminal-line-success"><?php echo esc_html($site_url); ?></span></span>
                <?php endif; ?>
                <?php if ($points_deducted > 0): ?>
                    <span class="terminal-line terminal-line-muted">تم خصم <?php echo intval($points_deducted); ?> نقطة من رصيدك لإنشاء الموقع.</span>
                <?php endif; ?>
                <span class="terminal-line terminal-line-muted">
                    تقدر دلوقتي تضيف صفحاتك، صورك، ومنتجاتك وتربط كل حاجة بنظامك التسويقي.
                    <span class="terminal-cursor"></span>
                </span>
            </div>
        </div>

        <?php if ($site_url): ?>
        <div class="success-button">
            <a href="<?php echo esc_url($site_url); ?>" class="button1" target="_blank" rel="noopener noreferrer">
                <span>↗</span><span>اذهب إلى موقعك الآن</span>
            </a>
        </div>
        <?php endif; ?>
        <p class="success-note">من خلال لوحة Sity Cloud تقدر تدير مواقعك، نقاطك، والباقات المفعّلة في مكان واحد.</p>
    </div>
    <script>
    setTimeout(function() {
        window.location.href = 'https://sity.cloud/mysites';
    }, 5000);
    </script>
    <?php
    return ob_get_clean();
}

/**
 * ==============
 * Sity Cloud Enhancements:
 * - إدارة مستخدمين إضافيين لكل موقع
 * - إعدادات نظام الإشعارات
 * - مراقبة النقاط والإشعارات البريدية
 * - اختبارات أداء تلقائية للمواقع
 * ==============
 */
add_action('admin_menu', 'sitycloud_register_extra_pages');
function sitycloud_register_extra_pages() {
    add_submenu_page(
        'fawry-points-dashboard',
        __('إدارة المستخدمين الإضافيين', 'fawry-site'),
        __('المستخدمون الإضافيون', 'fawry-site'),
        'manage_options',
        'sitycloud-additional-users',
        'sitycloud_additional_users_page'
    );

    add_submenu_page(
        'fawry-points-dashboard',
        __('نظام الإشعارات', 'fawry-site'),
        __('نظام الإشعارات', 'fawry-site'),
        'manage_options',
        'sitycloud-notifications',
        'sitycloud_notifications_page'
    );
}

function sitycloud_additional_users_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('غير مسموح لك بالوصول إلى هذه الصفحة.', 'fawry-site'));
    }

    $message = '';
    if (isset($_POST['sitycloud_add_user_nonce']) && wp_verify_nonce($_POST['sitycloud_add_user_nonce'], 'sitycloud_add_user')) {
        $site_id = isset($_POST['site_id']) ? intval($_POST['site_id']) : 0;
        $email   = isset($_POST['user_email']) ? sanitize_email($_POST['user_email']) : '';
        $name    = isset($_POST['user_name']) ? sanitize_text_field($_POST['user_name']) : '';
        $role    = isset($_POST['user_role']) ? sanitize_text_field($_POST['user_role']) : 'editor';

        if ($site_id && $email) {
            $user = get_user_by('email', $email);
            if (!$user) {
                $password = wp_generate_password(12, true);
                $username = sanitize_user(current(explode('@', $email)));
                $user_id  = wp_create_user($username, $password, $email);
                if (is_wp_error($user_id)) {
                    $message = __('تعذر إنشاء المستخدم الجديد. يرجى التحقق من البيانات.', 'fawry-site');
                } else {
                    if (!empty($name)) {
                        wp_update_user(array(
                            'ID'           => $user_id,
                            'display_name' => $name,
                        ));
                    }
                    $user = get_user_by('id', $user_id);
                }
            }

            if ($user && !is_wp_error($user)) {
                if (is_multisite()) {
                    add_user_to_blog($site_id, $user->ID, $role);
                } else {
                    $user->set_role($role);
                }
                $message = __('تم إضافة المستخدم الإضافي بنجاح إلى الموقع المحدد.', 'fawry-site');
            }
        } else {
            $message = __('يرجى اختيار الموقع وكتابة بريد المستخدم.', 'fawry-site');
        }
    }

    $sites = array();
    if (is_multisite()) {
        $sites = get_sites(array(
            'public'   => 1,
            'archived' => 0,
            'deleted'  => 0,
        ));
    }
    ?>
    <div class="wrap sitycloud-admin-page">
        <h1><?php _e('إدارة المستخدمين الإضافيين', 'fawry-site'); ?></h1>

        <?php if ($message): ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html($message); ?></p></div>
        <?php endif; ?>

        <form method="post" class="sitycloud-card">
            <?php wp_nonce_field('sitycloud_add_user', 'sitycloud_add_user_nonce'); ?>
            <h2><?php _e('إضافة مستخدم جديد إلى موقع', 'fawry-site'); ?></h2>

            <table class="form-table">
                <tr>
                    <th><label for="site_id"><?php _e('اختر الموقع', 'fawry-site'); ?></label></th>
                    <td>
                        <?php if ($sites) : ?>
                            <select name="site_id" id="site_id">
                                <?php foreach ($sites as $site) :
                                    $url  = get_home_url($site->blog_id, '/');
                                    $name = get_blog_option($site->blog_id, 'blogname');
                                ?>
                                    <option value="<?php echo esc_attr($site->blog_id); ?>"><?php echo esc_html($name . ' (' . $url . ')'); ?></option>
                                <?php endforeach; ?>
                            </select>
                        <?php else : ?>
                            <p><?php _e('لا توجد مواقع مسجلة على هذه الشبكة.', 'fawry-site'); ?></p>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th><label for="user_name"><?php _e('اسم المستخدم', 'fawry-site'); ?></label></th>
                    <td><input type="text" name="user_name" id="user_name" class="regular-text" /></td>
                </tr>
                <tr>
                    <th><label for="user_email"><?php _e('البريد الإلكتروني', 'fawry-site'); ?></label></th>
                    <td><input type="email" name="user_email" id="user_email" class="regular-text" required /></td>
                </tr>
                <tr>
                    <th><label for="user_role"><?php _e('دور المستخدم', 'fawry-site'); ?></label></th>
                    <td>
                        <select name="user_role" id="user_role">
                            <option value="administrator"><?php _e('مدير', 'fawry-site'); ?></option>
                            <option value="editor"><?php _e('محرر', 'fawry-site'); ?></option>
                            <option value="author"><?php _e('كاتب', 'fawry-site'); ?></option>
                        </select>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" class="button button-primary"><?php _e('إضافة المستخدم', 'fawry-site'); ?></button>
            </p>
        </form>
    </div>
    <?php
}

function sitycloud_notifications_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('غير مسموح لك بالوصول إلى هذه الصفحة.', 'fawry-site'));
    }

    if (isset($_POST['sitycloud_notifications_nonce']) && wp_verify_nonce($_POST['sitycloud_notifications_nonce'], 'sitycloud_notifications_save')) {
        $threshold  = isset($_POST['low_points_threshold']) ? absint($_POST['low_points_threshold']) : 10;
        $promo_text = isset($_POST['promo_message']) ? wp_kses_post($_POST['promo_message']) : '';
        update_option('sitycloud_low_points_threshold', $threshold);
        update_option('sitycloud_promo_message', $promo_text);
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('تم حفظ الإعدادات.', 'fawry-site') . '</p></div>';
    }

    $threshold  = get_option('sitycloud_low_points_threshold', 10);
    $promo_text = get_option('sitycloud_promo_message', '');
    ?>
    <div class="wrap sitycloud-admin-page">
        <h1><?php _e('إعدادات نظام الإشعارات', 'fawry-site'); ?></h1>

        <form method="post" class="sitycloud-card">
            <?php wp_nonce_field('sitycloud_notifications_save', 'sitycloud_notifications_nonce'); ?>

            <table class="form-table">
                <tr>
                    <th><label for="low_points_threshold"><?php _e('حد تنبيه انخفاض النقاط', 'fawry-site'); ?></label></th>
                    <td>
                        <input type="number" name="low_points_threshold" id="low_points_threshold" value="<?php echo esc_attr($threshold); ?>" min="0" />
                        <p class="description"><?php _e('سيتم تنبيه العميل إذا انخفض رصيد نقاطه عن هذا الحد.', 'fawry-site'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th><label for="promo_message"><?php _e('رسالة ترويجية عامة', 'fawry-site'); ?></label></th>
                    <td>
                        <textarea name="promo_message" id="promo_message" rows="5" class="large-text"><?php echo esc_textarea($promo_text); ?></textarea>
                        <p class="description"><?php _e('ستظهر هذه الرسالة للعملاء داخل لوحة التحكم ويمكن استخدامها للعروض أو التحديثات.', 'fawry-site'); ?></p>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" class="button button-primary"><?php _e('حفظ الإعدادات', 'fawry-site'); ?></button>
            </p>
        </form>
    </div>
    <?php
}

/**
 * مراقبة النقاط وإرسال تنبيهات عند الانخفاض
 */
add_action('updated_user_meta', 'sitycloud_watch_points_meta', 10, 4);
add_action('added_user_meta',   'sitycloud_watch_points_meta', 10, 4);
function sitycloud_watch_points_meta($meta_id, $user_id, $meta_key, $_meta_value) {
    if ($meta_key !== 'fawry_points') {
        return;
    }
    $points = intval(get_user_meta($user_id, 'fawry_points', true));
    sitycloud_maybe_notify_low_points($user_id, $points);
}

function sitycloud_maybe_notify_low_points($user_id, $points) {
    $threshold = intval(get_option('sitycloud_low_points_threshold', 10));
    if ($threshold <= 0) {
        return;
    }

    if ($points < $threshold) {
        $user = get_userdata($user_id);
        if (!$user) return;

        $subject = __('تنبيه: رصيد نقاطك على Sity Cloud منخفض', 'fawry-site');
        $message = sprintf(
            __("مرحبًا %s,\n\nرصيد نقاطك الحالي على Sity Cloud هو: %d نقطة.\nننصحك بشحن رصيدك للاستمرار في إنشاء المواقع واستخدام خدماتك بدون انقطاع.\n\nتحيات فريق Sity Cloud.", 'fawry-site'),
            $user->display_name,
            $points
        );

        wp_mail($user->user_email, $subject, $message);
    }
}

/**
 * إشعارات داخل لوحة التحكم (لو / promo)
 */
add_action('admin_notices', 'sitycloud_show_dashboard_notices');
function sitycloud_show_dashboard_notices() {
    if (!is_user_logged_in()) return;

    $user_id   = get_current_user_id();
    $points    = intval(get_user_meta($user_id, 'fawry_points', true));
    $threshold = intval(get_option('sitycloud_low_points_threshold', 10));
    $promo     = get_option('sitycloud_promo_message', '');

    if ($threshold > 0 && $points > 0 && $points < $threshold) {
        echo '<div class="notice notice-warning sitycloud-notice"><p>' .
            esc_html(sprintf(__('تنبيه: رصيد نقاطك الحالي هو %d نقطة، وقد يقترب من النفاد.', 'fawry-site'), $points)) .
            '</p></div>';
    }

    if (!empty($promo) && current_user_can('read')) {
        echo '<div class="notice notice-info sitycloud-notice"><p>' . wp_kses_post($promo) . '</p></div>';
    }
}


/**
 * إنشاء جدول مركزي لتخزين بيانات المواقع والمشروعات التي يتم إنشاؤها من Sity Cloud
 */
function fawry_create_sites_info_table() {
    global $wpdb;

    $table_name      = $wpdb->base_prefix . 'sitycloud_sites_info';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        site_id bigint(20) unsigned NOT NULL,
        user_id bigint(20) unsigned NOT NULL,
        site_url varchar(255) NOT NULL,
        site_name varchar(255) NOT NULL,
        phone_number varchar(50) DEFAULT '',
        country varchar(100) DEFAULT '',
        city varchar(100) DEFAULT '',
        whatsapp_link varchar(255) DEFAULT '',
        client_type varchar(50) DEFAULT '',
        company_name varchar(255) DEFAULT '',
        business_registration_number varchar(255) DEFAULT '',
        industry varchar(255) DEFAULT '',
        activity_type varchar(255) DEFAULT '',
        referral_source varchar(255) DEFAULT '',
        points_payment_method varchar(255) DEFAULT '',
        project_description longtext,
        marketing_budget varchar(255) DEFAULT '',
        products_description longtext,
        website_goal longtext,
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        KEY site_id (site_id),
        KEY user_id (user_id)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
}

/**
 * حفظ بيانات مشروع الموقع في الجدول المركزي لسهولة المتابعة والتصدير من لوحة تحكم الشبكة
 */
function fawry_store_site_project_info($site_id, $user_id, $data) {
    global $wpdb;

    $table_name = $wpdb->base_prefix . 'sitycloud_sites_info';

    // تأكد من وجود الجدول (في حال تم تحديث البلجن بدون إعادة تفعيل)
    $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));
    if ($table_exists !== $table_name) {
        fawry_create_sites_info_table();
    }

    $defaults = array(
        'site_url'                     => '',
        'site_name'                    => '',
        'phone_number'                 => '',
        'country'                      => '',
        'city'                         => '',
        'whatsapp_link'                => '',
        'client_type'                  => '',
        'company_name'                 => '',
        'business_registration_number' => '',
        'industry'                     => '',
        'activity_type'                => '',
        'referral_source'              => '',
        'points_payment_method'        => '',
        'project_description'          => '',
        'marketing_budget'             => '',
        'products_description'         => '',
        'website_goal'                 => '',
    );

    $data = wp_parse_args($data, $defaults);

    $wpdb->insert(
        $table_name,
        array(
            'site_id'                     => (int) $site_id,
            'user_id'                     => (int) $user_id,
            'site_url'                    => $data['site_url'],
            'site_name'                   => $data['site_name'],
            'phone_number'                => $data['phone_number'],
            'country'                     => $data['country'],
            'city'                        => $data['city'],
            'whatsapp_link'               => $data['whatsapp_link'],
            'client_type'                 => $data['client_type'],
            'company_name'                => $data['company_name'],
            'business_registration_number'=> $data['business_registration_number'],
            'industry'                    => $data['industry'],
            'activity_type'               => $data['activity_type'],
            'referral_source'             => $data['referral_source'],
            'points_payment_method'       => $data['points_payment_method'],
            'project_description'         => $data['project_description'],
            'marketing_budget'            => $data['marketing_budget'],
            'products_description'        => $data['products_description'],
            'website_goal'                => $data['website_goal'],
            'created_at'                  => current_time('mysql'),
        ),
        array(
            '%d', // site_id
            '%d', // user_id
            '%s', // site_url
            '%s', // site_name
            '%s', // phone_number
            '%s', // country
            '%s', // city
            '%s', // whatsapp_link
            '%s', // client_type
            '%s', // company_name
            '%s', // business_registration_number
            '%s', // industry
            '%s', // activity_type
            '%s', // referral_source
            '%s', // points_payment_method
            '%s', // project_description
            '%s', // marketing_budget
            '%s', // products_description
            '%s', // website_goal
            '%s', // created_at
        )
    );
}

/**
 * اختبارات أداء تلقائية (كرون يومي)
 */
register_activation_hook(__FILE__, 'sitycloud_plugin_activate');
function sitycloud_plugin_activate() {
    if (!wp_next_scheduled('sitycloud_performance_test_event')) {
        wp_schedule_event(time(), 'daily', 'sitycloud_performance_test_event');
    }

    // إنشاء جدول بيانات المواقع / المشاريع في لوحة تحكم الشبكة
    fawry_create_sites_info_table();
}
register_deactivation_hook(__FILE__, 'sitycloud_plugin_deactivate');
function sitycloud_plugin_deactivate() {
    wp_clear_scheduled_hook('sitycloud_performance_test_event');
}

add_action('sitycloud_performance_test_event', 'sitycloud_run_performance_tests');
function sitycloud_run_performance_tests() {
    if (!is_multisite()) return;

    $sites = get_sites(array(
        'public'   => 1,
        'archived' => 0,
        'deleted'  => 0,
    ));

    foreach ($sites as $site) {
        $blog_id = $site->blog_id;
        $url     = get_home_url($blog_id, '/');

        $start    = microtime(true);
        $response = wp_remote_get($url, array('timeout' => 15));
        $time     = microtime(true) - $start;
        $code     = wp_remote_retrieve_response_code($response);

        if (is_wp_error($response) || $code !== 200) {
            $score = 0;
        } else {
            $score = max(0, min(100, 100 - ($time * 100)));
        }

        update_blog_option($blog_id, 'sitycloud_last_performance_score', $score);
        update_blog_option($blog_id, 'sitycloud_last_performance_time', $time);
        update_blog_option($blog_id, 'sitycloud_last_performance_checked', current_time('mysql'));

        if ($score > 0 && $score < 40) {
            $admin_email = get_blog_option($blog_id, 'admin_email');
            if ($admin_email) {
                $subject = __('تقرير أداء موقعك على Sity Cloud', 'fawry-site');
                $message = sprintf(
                    __("مرحبًا،\n\nتم قياس سرعة موقعك (%s) وكان التقييم التقريبي للأداء: %d/100 (وقت استجابة: %.2f ثانية).\nننصحك بمراجعة أداء موقعك وتحسين السرعة لتحسين تجربة الزوار.\n\nتحيات فريق Sity Cloud.", 'fawry-site'),
                    $url,
                    intval($score),
                    $time
                );
                wp_mail($admin_email, $subject, $message);
            }
        }
    }
}

/**
 * إظهار نتيجة الأداء في شريط الأدمن
 */
add_action('admin_bar_menu', 'sitycloud_adminbar_performance', 100);
function sitycloud_adminbar_performance($wp_admin_bar) {
    if (!is_user_logged_in() || !is_super_admin() || !is_multisite()) return;

    $blog_id = get_current_blog_id();
    $score   = get_blog_option($blog_id, 'sitycloud_last_performance_score', '');
    $label   = $score !== '' ? sprintf(__('أداء الموقع: %s/100', 'fawry-site'), $score) : __('لم يتم قياس الأداء بعد', 'fawry-site');

    $wp_admin_bar->add_node(array(
        'id'    => 'sitycloud-performance',
        'title' => esc_html($label),
        'href'  => admin_url('admin.php?page=fawry-points-dashboard'),
    ));
}

/**
 * تتبع إصدار البلجن للإشعار بعد التحديث
 */
function sitycloud_get_plugin_version() {
    $data = get_file_data(__FILE__, array('Version' => 'Version'), 'plugin');
    return isset($data['Version']) ? $data['Version'] : '';
}
add_action('admin_init', 'sitycloud_maybe_show_update_notice');
function sitycloud_maybe_show_update_notice() {
    $stored_version = get_option('sitycloud_plugin_version', '');
    $current        = sitycloud_get_plugin_version();
    if ($current && $current !== $stored_version) {
        update_option('sitycloud_plugin_version', $current);
        add_action('admin_notices', function () {
            echo '<div class="notice notice-success is-dismissible sitycloud-notice"><p>';
            echo esc_html__('تم تحديث إضافة Sity Cloud بنجاح – راجع الميزات الجديدة في لوحة النقاط.', 'fawry-site');
            echo '</p></div>';
        });
    }
}

/**
 * CSS + JS للتصميم الزجاجي + الويزارد + أيقونة الإشعارات + نافذة لوحة التحكم
 * لا يوجد Dark Mode – لايت فقط
 */
function fawry_enqueue_styles() {
    ?>
    <style>
    :root {
        --sity-bg: #f4f3ff;
        --sity-glass-bg: rgba(255, 255, 255, 0.9);
        --sity-glass-border: rgba(255, 255, 255, 0.6);
        --sity-accent: #7C3AED;
        --sity-accent-soft: rgba(124, 58, 237, 0.12);
        --sity-accent-strong: #5B21B6;
        --sity-text-main: #0f172a;
        --sity-text-muted: #64748b;
        --sity-radius-lg: 18px;
        --sity-shadow-soft: 0 18px 40px rgba(15, 23, 42, 0.16);
        --sity-transition: 0.25s ease;
    }

    .form_container,
    #success-page,
    .fawry-card,
    .sitycloud-card {
        background: radial-gradient(circle at top left, rgba(124, 58, 237, 0.12), transparent 55%), var(--sity-glass-bg);
        border-radius: var(--sity-radius-lg);
        border: 1px solid var(--sity-glass-border);
        box-shadow: var(--sity-shadow-soft);
        backdrop-filter: blur(16px);
        -webkit-backdrop-filter: blur(16px);
        position: relative;
        overflow: hidden;
    }

    .form_container {
        max-width: 960px;
        margin: 32px auto;
        padding: 28px 24px 24px;
        display: flex;
        flex-direction: column;
        gap: 18px;
    }

    .title_container {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        gap: 16px;
        flex-wrap: wrap;
    }

    .title_container .title {
        font-size: 1.6rem;
        font-weight: 800;
        letter-spacing: 0.02em;
        color: var(--sity-text-main);
        margin: 0;
    }

    .title_container .subtitle {
        font-size: 0.9rem;
        color: var(--sity-text-muted);
    }

    .input_container {
        display: flex;
        flex-direction: column;
        gap: 6px;
        margin-bottom: 8px;
    }

    .input_label {
        font-size: 0.9rem;
        font-weight: 600;
        color: var(--sity-text-muted);
        display: flex;
        align-items: center;
        gap: 6px;
    }

    .input_label::before {
        content: "";
        width: 6px;
        height: 6px;
        border-radius: 999px;
        background: linear-gradient(135deg, var(--sity-accent), var(--sity-accent-strong));
    }

    .input_field,
    .form_container input[type="text"],
    .form_container input[type="email"],
    .form_container input[type="tel"],
    .form_container input[type="url"],
    .form_container select,
    .form_container textarea {
        border-radius: 12px;
        border: 1px solid rgba(148, 163, 184, 0.5);
        padding: 9px 11px;
        font-size: 0.95rem;
        background: rgba(255, 255, 255, 0.98);
        color: var(--sity-text-main);
        outline: none;
        width: 100%;
        transition: border-color var(--sity-transition), box-shadow var(--sity-transition), transform var(--sity-transition);
    }

    .input_field:focus,
    .form_container input:focus,
    .form_container select:focus,
    .form_container textarea:focus {
        border-color: var(--sity-accent);
        box-shadow: 0 0 0 1px rgba(124, 58, 237, 0.28);
        transform: translateY(-1px);
    }

    .input_field.sity-field-error,
    .form_container .sity-field-error {
        border-color: #ef4444;
        box-shadow: 0 0 0 1px rgba(239, 68, 68, 0.35);
    }

    .field_hint {
        font-size: 0.78rem;
        color: var(--sity-text-muted);
    }

    .url_prefix {
        padding: 8px 12px;
        border-radius: 999px;
        background: var(--sity-accent-soft);
        color: var(--sity-accent-strong);
        font-size: 0.8rem;
        white-space: nowrap;
    }

    .input_field_container {
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .radio_container {
        display: inline-flex;
        flex-wrap: wrap;
        gap: 8px;
        padding: 4px;
        border-radius: 999px;
        background: rgba(148, 163, 184, 0.16);
    }

    .radio_option {
        position: relative;
        cursor: pointer;
        font-size: 0.85rem;
        color: var(--sity-text-muted);
    }

    .radio_option input {
        display: none;
    }

    .radio_option span {
        display: inline-block;
        padding: 5px 13px;
        border-radius: 999px;
        transition: background var(--sity-transition), color var(--sity-transition), box-shadow var(--sity-transition);
    }

    .radio_option input:checked + span {
        background: linear-gradient(135deg, var(--sity-accent), var(--sity-accent-strong));
        color: #fff;
        box-shadow: 0 10px 25px rgba(124, 58, 237, 0.35);
    }

    .checkbox_container {
        display: flex;
        align-items: flex-start;
        gap: 8px;
        font-size: 0.85rem;
        color: var(--sity-text-muted);
        margin-top: 8px;
    }

    .checkbox_container input[type="checkbox"] {
        margin-top: 2px;
        width: 17px;
        height: 17px;
        border-radius: 5px;
        border: 1px solid rgba(148, 163, 184, 0.7);
    }

    .checkbox_container a {
        color: var(--sity-accent-strong);
        text-decoration: none;
        font-weight: 600;
    }

    .checkbox_container a:hover {
        text-decoration: underline;
    }

    .button1 {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 6px;
        border-radius: 999px;
        border: none;
        padding: 9px 18px;
        font-size: 0.9rem;
        font-weight: 600;
        cursor: pointer;
        background: linear-gradient(135deg, var(--sity-accent), var(--sity-accent-strong));
        color: #fff;
        box-shadow: 0 14px 32px rgba(124, 58, 237, 0.5);
        text-decoration: none;
        transition: transform var(--sity-transition), box-shadow var(--sity-transition), filter var(--sity-transition);
        white-space: nowrap;
    }

    .button1 span:first-child {
        font-size: 1.1rem;
    }

    .button1:hover {
        transform: translateY(-1px);
        box-shadow: 0 18px 40px rgba(124, 58, 237, 0.6);
        filter: brightness(1.04);
    }

    .button1:active {
        transform: translateY(0);
        box-shadow: 0 10px 24px rgba(124, 58, 237, 0.45);
    }

    .fawry-site-action {
        padding-inline: 14px;
        font-size: 0.82rem;
        box-shadow: 0 10px 20px rgba(124, 58, 237, 0.35);
    }

    .fawry-site-delete {
        background: linear-gradient(135deg, #ef4444, #b91c1c);
        box-shadow: 0 10px 20px rgba(239, 68, 68, 0.5);
    }

    .wizard_buttons {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-top: 12px;
    }

    .wizard_buttons_right {
        display: flex;
        gap: 8px;
    }

    .sity-step-indicator {
        display: flex;
        justify-content: space-between;
        gap: 8px;
        padding: 10px 12px;
        margin-bottom: 6px;
        border-radius: 999px;
        background: rgba(148, 163, 184, 0.14);
    }

    .sity-step-indicator-item {
        flex: 1;
        display: flex;
        align-items: center;
        gap: 8px;
        padding: 4px 8px;
        border-radius: 999px;
        color: var(--sity-text-muted);
        font-size: 0.8rem;
        transition: background var(--sity-transition), color var(--sity-transition), transform var(--sity-transition);
    }

    .sity-step-indicator-item.sity-step-active {
        background: linear-gradient(135deg, var(--sity-accent-soft), rgba(255, 255, 255, 0.9));
        color: var(--sity-accent-strong);
        transform: translateY(-1px);
    }

    .sity-step-number {
        width: 22px;
        height: 22px;
        border-radius: 999px;
        background: #fff;
        border: 1px solid rgba(148, 163, 184, 0.6);
        display: inline-flex;
        align-items: center;
        justify-content: center;
        font-size: 0.78rem;
        font-weight: 700;
        color: var(--sity-accent-strong);
    }

    .sity-step-label {
        flex: 1;
    }

    .sity-step-info {
        font-size: 0.9rem;
        color: var(--sity-text-muted);
        margin-bottom: 6px;
    }

    .sity-step-grid-2 {
        display: grid;
        grid-template-columns: repeat(2, minmax(0, 1fr));
        gap: 10px 14px;
    }

    .sity-alert {
        border-radius: 12px;
        padding: 10px 12px;
        font-size: 0.85rem;
        margin-bottom: 10px;
    }

    .sity-alert-error {
        background: rgba(248, 113, 113, 0.1);
        border: 1px solid rgba(239, 68, 68, 0.5);
        color: #b91c1c;
    }

    /* كروت المواقع */
    .fawry-cards {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
        gap: 18px;
        margin: 28px auto;
    }

    .fawry-card {
        padding: 0;
        text-align: left;
        display: flex;
        flex-direction: column;
        overflow: hidden;
    }

    .fawry-card-thumb {
        position: relative;
        overflow: hidden;
        max-height: 210px;
    }

    .fawry-card-thumb img {
        width: 100%;
        display: block;
        object-fit: cover;
        transform: scale(1.02);
        transition: transform 0.45s ease, filter 0.45s ease;
        filter: saturate(1.1);
    }

    .fawry-card:hover .fawry-card-thumb img {
        transform: scale(1.08);
        filter: saturate(1.2);
    }

    .fawry-card-inner {
        padding: 14px 14px 12px;
    }

    .fawry-card-header-row {
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 8px;
        margin-bottom: 4px;
    }

    .fawry-card h3 {
        margin: 4px 0 4px;
        font-size: 1.05rem;
        font-weight: 700;
        color: var(--sity-text-main);
    }

    .fawry-card h3 a {
        color: inherit;
        text-decoration: none;
    }

    .fawry-card h3 a:hover {
        color: var(--sity-accent-strong);
    }

    .fawry-card-meta {
        font-size: 0.8rem;
        color: var(--sity-text-muted);
        margin-bottom: 8px;
    }

    .fawry-status-pill {
        display: inline-flex;
        align-items: center;
        padding: 2px 8px;
        border-radius: 999px;
        font-size: 0.75rem;
        font-weight: 600;
    }

    .fawry-status-pill-active {
        background: var(--sity-accent-soft);
        color: var(--sity-accent-strong);
    }

    .fawry-status-pill-paused {
        background: rgba(148, 163, 184, 0.2);
        color: #111827;
    }

    .fawry-status-pill-delete {
        background: rgba(239, 68, 68, 0.12);
        color: #b91c1c;
    }

    .fawry-iban-pill {
        display: inline-flex;
        align-items: center;
        padding: 2px 8px;
        border-radius: 999px;
        background: rgba(15, 23, 42, 0.03);
        border: 1px dashed rgba(148, 163, 184, 0.6);
        font-size: 0.75rem;
        color: var(--sity-text-muted);
        max-width: 55%;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }

    .fawry-card-chips {
        display: flex;
        flex-wrap: wrap;
        gap: 6px;
        margin-bottom: 10px;
    }

    .fawry-chip {
        font-size: 0.78rem;
        padding: 3px 8px;
        border-radius: 999px;
        background: rgba(148, 163, 184, 0.14);
        color: var(--sity-text-muted);
    }

    .fawry-chip-cost {
        background: rgba(124, 58, 237, 0.12);
        color: var(--sity-accent-strong);
        font-weight: 600;
    }

    .fawry-card-actions {
        display: flex;
        flex-wrap: wrap;
        gap: 8px;
        margin-top: 4px;
    }

    .fawry-card-form {
        display: inline-flex;
        flex-wrap: wrap;
        gap: 6px;
        margin: 0;
    }

    /* صفحة النجاح */
    #success-page {
        max-width: 880px;
        margin: 32px auto;
        padding: 26px 22px 20px;
        text-align: center;
    }

    #success-page h1 {
        font-size: 1.9rem;
        margin: 0 0 6px;
        background: linear-gradient(135deg, var(--sity-accent), var(--sity-accent-strong));
        -webkit-background-clip: text;
        background-clip: text;
        color: transparent;
    }

    #success-page .subtitle {
        font-size: 0.9rem;
        color: var(--sity-text-muted);
    }

    .terminal-loader {
        margin: 20px auto 14px;
        max-width: 540px;
        border-radius: 14px;
        background: #020617;
        padding: 14px 12px 10px;
        color: #e5e7eb;
        box-shadow: 0 18px 40px rgba(15, 23, 42, 0.8);
        font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
    }

    .terminal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 6px;
    }

    .terminal-title {
        font-size: 0.8rem;
        color: #9ca3af;
    }

    .terminal-buttons span {
        display: inline-block;
        width: 8px;
        height: 8px;
        border-radius: 999px;
        background: #4b5563;
        margin-left: 4px;
    }

    .terminal-body {
        font-size: 0.8rem;
        line-height: 1.5;
    }

    .terminal-line {
        display: block;
        margin-bottom: 2px;
    }

    .terminal-line-prefix {
        color: #6b7280;
        margin-right: 4px;
    }

    .terminal-line-success {
        color: #22c55e;
    }

    .terminal-line-muted {
        color: #9ca3af;
    }

    .terminal-cursor {
        display: inline-block;
        width: 6px;
        height: 11px;
        background: #22c55e;
        margin-left: 4px;
        animation: sity-cursor 1s infinite;
    }

    .success-button {
        margin-top: 18px;
    }

    .success-note {
        font-size: 0.85rem;
        color: var(--sity-text-muted);
        margin-top: 6px;
    }

    /* أيقونة الإشعارات */
    .sity-notification-bell {
        width: 34px;
        height: 34px;
        border-radius: 999px;
        border: 1px solid rgba(148, 163, 184, 0.5);
        background: rgba(255, 255, 255, 0.9);
        display: inline-flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        box-shadow: 0 10px 24px rgba(15, 23, 42, 0.15);
        transition: transform var(--sity-transition), box-shadow var(--sity-transition), background var(--sity-transition);
        position: relative;
    }

    .sity-notification-bell::before {
        content: "🔔";
        font-size: 1.1rem;
    }

    .sity-notification-bell:hover {
        transform: translateY(-1px);
        box-shadow: 0 14px 32px rgba(15, 23, 42, 0.2);
        background: rgba(255, 255, 255, 1);
    }

    /* Toast notifications */
    .sity-toast-container {
        position: fixed;
        top: 16px;
        left: 50%;
        transform: translateX(-50%);
        z-index: 99999;
        display: flex;
        flex-direction: column;
        gap: 8px;
        max-width: 380px;
        width: calc(100% - 24px);
        pointer-events: none;
    }

    .sity-toast {
        pointer-events: auto;
        border-radius: 14px;
        padding: 10px 12px;
        font-size: 0.85rem;
        display: flex;
        align-items: flex-start;
        gap: 8px;
        background: rgba(255, 255, 255, 0.98);
        border: 1px solid rgba(148, 163, 184, 0.6);
        box-shadow: 0 12px 30px rgba(15, 23, 42, 0.25);
        direction: rtl;
    }

    .sity-toast-icon {
        margin-top: 2px;
        font-size: 1.1rem;
    }

    .sity-toast-message {
        flex: 1;
        color: var(--sity-text-main);
    }

    .sity-toast-close {
        border: none;
        background: transparent;
        cursor: pointer;
        font-size: 1rem;
        line-height: 1;
        color: var(--sity-text-muted);
    }

    .sity-toast-info {
        border-right: 3px solid var(--sity-accent);
    }

    .sity-toast-error {
        border-right: 3px solid #ef4444;
    }

    .sity-toast-success {
        border-right: 3px solid #22c55e;
    }

    /* admin cards */
    .sitycloud-admin-page .sitycloud-card {
        margin-top: 14px;
        padding: 18px 18px 16px;
    }

    .sitycloud-notice p {
        font-size: 0.9rem;
    }

    /* نافذة لوحة التحكم (modal) */
    .sity-admin-modal-backdrop {
        position: fixed;
        inset: 0;
        background: rgba(15, 23, 42, 0.8);
        z-index: 99998;
        display: none;
        align-items: stretch;
        justify-content: stretch;
    }

    .sity-admin-modal {
        flex: 1;
        margin: 10px;
        border-radius: 16px;
        background: #020617;
        box-shadow: 0 24px 60px rgba(0, 0, 0, 0.75);
        position: relative;
        overflow: hidden;
        display: flex;
        flex-direction: column;
    }

    .sity-admin-modal-header {
        height: 42px;
        display: flex;
        justify-content: flex-end;
        align-items: center;
        padding: 0 10px;
        background: linear-gradient(135deg, rgba(15,23,42,0.96), rgba(30,64,175,0.9));
        color: #e5e7eb;
    }

    .sity-admin-modal-close {
        border: none;
        background: transparent;
        color: inherit;
        cursor: pointer;
        font-size: 1.2rem;
        width: 32px;
        height: 32px;
        border-radius: 999px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        transition: background 0.2s;
    }

    .sity-admin-modal-close:hover {
        background: rgba(15, 23, 42, 0.5);
    }

    .sity-admin-modal-body {
        flex: 1;
        background: #020617;
    }

    .sity-admin-modal-body iframe {
        width: 100%;
        height: 100%;
        border: none;
    }

    @keyframes sity-cursor {
        0%, 49% { opacity: 1; }
        50%, 100% { opacity: 0; }
    }

    @media (max-width: 768px) {
        .form_container {
            margin: 20px 12px;
            padding: 20px 14px;
        }
        .sity-step-grid-2 {
            grid-template-columns: 1fr;
        }
        .wizard_buttons {
            flex-direction: column;
            align-items: stretch;
            gap: 8px;
        }
        .wizard_buttons_right {
            justify-content: space-between;
        }
    }
    </style>
    <script>
    (function() {

        // Toast Notifications
        function SityCloudShowNotification(message, type) {
            type = type || 'info';
            var container = document.querySelector('.sity-toast-container');
            if (!container) {
                container = document.createElement('div');
                container.className = 'sity-toast-container';
                document.body.appendChild(container);
            }

            var toast = document.createElement('div');
            toast.className = 'sity-toast sity-toast-' + type;

            var icon = document.createElement('div');
            icon.className = 'sity-toast-icon';

            if (type === 'error') {
                icon.textContent = '⚠️';
            } else if (type === 'success') {
                icon.textContent = '✅';
            } else {
                icon.textContent = '🔔';
            }

            var msg = document.createElement('div');
            msg.className = 'sity-toast-message';
            msg.textContent = message || '';

            var close = document.createElement('button');
            close.className = 'sity-toast-close';
            close.type = 'button';
            close.innerHTML = '&times;';
            close.addEventListener('click', function() {
                if (toast && toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            });

            toast.appendChild(icon);
            toast.appendChild(msg);
            toast.appendChild(close);
            container.appendChild(toast);

            setTimeout(function() {
                if (toast && toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 4500);
        }
        window.SityCloudShowNotification = SityCloudShowNotification;

        // Wizard logic
        function initWizard() {
            var form = document.getElementById('fawry-create-site-form');
            if (!form) return;

            var steps        = form.querySelectorAll('.sity-step');
            if (!steps.length) return;

            var currentIndex = 0;
            var nextBtn      = form.querySelector('.sity-step-next');
            var prevBtn      = form.querySelector('.sity-step-prev');
            var submitBtn    = form.querySelector('.sity-step-submit');
            var indicators   = form.querySelectorAll('.sity-step-indicator-item');

            function setRequiredForStep(index) {
                var allRequired = form.querySelectorAll('[data-required="1"]');
                allRequired.forEach(function(field) {
                    field.removeAttribute('required');
                });

                var activeStep = steps[index];
                var stepFields = activeStep.querySelectorAll('[data-required="1"]');
                stepFields.forEach(function(field) {
                    field.setAttribute('required', 'required');
                });
            }

            function validateStep(index) {
                var step   = steps[index];
                var fields = step.querySelectorAll('[data-required="1"]');
                var valid  = true;

                fields.forEach(function(field) {
                    field.classList.remove('sity-field-error');
                    if (field.type === 'checkbox') {
                        if (!field.checked) {
                            field.classList.add('sity-field-error');
                            valid = false;
                        }
                    } else if (!field.value || !field.value.trim()) {
                        field.classList.add('sity-field-error');
                        valid = false;
                    }
                });

                if (!valid) {
                    SityCloudShowNotification('من فضلك أكمل الحقول المطلوبة في هذه الخطوة قبل المتابعة.', 'error');
                }
                return valid;
            }

            function showStep(index) {
                steps.forEach(function(step, i) {
                    step.style.display = (i === index) ? 'block' : 'none';
                });

                indicators.forEach(function(el, i) {
                    if (i === index) {
                        el.classList.add('sity-step-active');
                    } else {
                        el.classList.remove('sity-step-active');
                    }
                });

                prevBtn.style.display   = (index === 0) ? 'none' : 'inline-flex';
                nextBtn.style.display   = (index === steps.length - 1) ? 'none' : 'inline-flex';
                submitBtn.style.display = (index === steps.length - 1) ? 'inline-flex' : 'none';

                currentIndex = index;
                setRequiredForStep(index);
            }

            if (nextBtn) {
                nextBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    if (!validateStep(currentIndex)) return;
                    if (currentIndex < steps.length - 1) {
                        showStep(currentIndex + 1);
                    }
                });
            }

            if (prevBtn) {
                prevBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    if (currentIndex > 0) {
                        showStep(currentIndex - 1);
                    }
                });
            }

            showStep(0);
        }

        // نافذة لوحة التحكم (modal) + فتح wp-admin داخل iframe
        var adminOverlay = null;
        var adminIframe  = null;

        function ensureAdminOverlay() {
            if (adminOverlay) return;

            adminOverlay = document.createElement('div');
            adminOverlay.className = 'sity-admin-modal-backdrop';

            var modal = document.createElement('div');
            modal.className = 'sity-admin-modal';

            var header = document.createElement('div');
            header.className = 'sity-admin-modal-header';

            var closeBtn = document.createElement('button');
            closeBtn.type = 'button';
            closeBtn.className = 'sity-admin-modal-close';
            closeBtn.innerHTML = '&times;';
            closeBtn.addEventListener('click', function() {
                adminOverlay.style.display = 'none';
                if (adminIframe) {
                    adminIframe.src = 'about:blank';
                }
            });

            header.appendChild(closeBtn);

            var body = document.createElement('div');
            body.className = 'sity-admin-modal-body';

            adminIframe = document.createElement('iframe');
            adminIframe.setAttribute('loading', 'lazy');
            body.appendChild(adminIframe);

            modal.appendChild(header);
            modal.appendChild(body);
            adminOverlay.appendChild(modal);
            document.body.appendChild(adminOverlay);
        }

        function SityCloudOpenAdminPanel(url) {
            if (!url) return;
            ensureAdminOverlay();
            adminIframe.src = url;
            adminOverlay.style.display = 'flex';
        }
        window.SityCloudOpenAdminPanel = SityCloudOpenAdminPanel;

        function bindAdminButtons() {
            var buttons = document.querySelectorAll('.sity-open-admin');
            if (!buttons.length) return;
            buttons.forEach(function(btn) {
                btn.addEventListener('click', function(e) {
                    e.preventDefault();
                    var url = btn.getAttribute('data-admin-url');
                    SityCloudOpenAdminPanel(url);
                });
            });

            // تأكيد حذف الموقع
            var deleteButtons = document.querySelectorAll('.fawry-site-delete');
            deleteButtons.forEach(function(btn) {
                btn.addEventListener('click', function(e) {
                    var ok = confirm('هل أنت متأكد من حذف هذا الموقع؟ لا يمكن التراجع عن هذا الإجراء.\nإذا لم تكن سوبر أدمن سيتم إرسال طلب حذف لفريق Sity Cloud.');
                    if (!ok) {
                        e.preventDefault();
                    }
                });
            });
        }

        function showSiteActionNotifications() {
            var params = new URLSearchParams(window.location.search);
            var res    = params.get('site_action_result');
            if (!res) return;

            if (res === 'paused') {
                SityCloudShowNotification('تم تعليق الموقع بنجاح. يمكنك إعادة تشغيله في أي وقت من نفس القائمة.', 'success');
            } else if (res === 'resumed') {
                SityCloudShowNotification('تم تشغيل الموقع مرة أخرى وهو الآن متاح للزوار.', 'success');
            } else if (res === 'deleted') {
                SityCloudShowNotification('تم حذف الموقع نهائيًا من شبكة Sity Cloud.', 'success');
            } else if (res === 'delete_requested') {
                SityCloudShowNotification('تم إرسال طلب حذف موقعك إلى فريق Sity Cloud. سيتم مراجعته والرد عليك.', 'info');
            } else if (res === 'no_permission') {
                SityCloudShowNotification('لا تملك صلاحية كافية لتنفيذ هذا الإجراء على الموقع.', 'error');
            } else if (res === 'error') {
                SityCloudShowNotification('حدث خطأ أثناء تنفيذ العملية على الموقع. برجاء المحاولة مرة أخرى.', 'error');
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            initWizard();
            bindAdminButtons();
            showSiteActionNotifications();
        });

    })();
    </script>
    <?php
}
add_action('wp_head', 'fawry_enqueue_styles');

?>
